<?php

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       http://wpautoroyalplugin.com
 * @since      1.0.2
 *
 * @package    autoroyal
 * @subpackage autoroyal/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.2
 * @package    autoroyal
 * @subpackage autoroyal/includes
 * @author     Your Name <email@example.com>
 */
class autoroyal {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.2
	 * @access   protected
	 * @var      autoroyal_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.2
	 * @access   protected
	 * @var      string    $autoroyal    The string used to uniquely identify this plugin.
	 */
	protected $autoroyal;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.2
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the autoroyal and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.2
	 */
	public function __construct() {

		$this->autoroyal = 'autoroyal';
		$this->version = '1.0.2';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();

		// Includes
		include( 'class-autoroyal-install.php' );

		include( 'class-autoroyal-car-post-types.php' );
		include( 'class-autoroyal-car-metaboxes.php' );

		// Custom Post Types
		include( 'post-types/class-autoroyal-team-post-type.php' );
		include( 'post-types/class-autoroyal-partners-post-type.php' );
		include( 'post-types/class-autoroyal-megamenu-post-type.php' );

		include( 'class-autoroyal-lead-post-types.php' );
		include( 'class-autoroyal-lead-metaboxes.php' );

		include( 'class-autoroyal-deal-post-types.php' );
		include( 'class-autoroyal-deal-metaboxes.php' );

		include( 'class-autoroyal-rent-post-types.php' );
		include( 'class-autoroyal-rent-metaboxes.php' );

		include( 'class-autoroyal-filter-post-types.php' );

		include( 'autoroyal-core-functions.php' );
		include( 'autoroyal-template.php' );

		include( 'class-autoroyal-geolocation.php' );

		// Forms
		include( 'forms/contact-request.php' );
		include( 'forms/pre-qualify-form.php' );
		include( 'forms/financial-application.php' );
		include( 'forms/trade-in-form.php' );
		include( 'forms/test-drive-form.php' );
		
		// Search Filters
		include( 'forms/search-horizontal-filter.php' );
		include( 'forms/search-horizontal-modern-filter.php' );
		include( 'forms/search-vertical-filter.php' );

		// Rent Filters
		include( 'forms/search-vertical-filter-rent.php' );

		// Shortcodes
		include( 'class-autoroyal-shortcode-dashboard.php' );
		include( 'class-autoroyal-shortcode-inventory.php' );
		include( 'class-autoroyal-shortcode-active-advertised-listings.php' );
		include( 'class-autoroyal-shortcode-submit.php' );
		include( 'class-autoroyal-shortcode-leads.php' );
		include( 'class-autoroyal-shortcode-submit-lead.php' );
		include( 'class-autoroyal-shortcode-deals.php' );
		include( 'class-autoroyal-shortcode-submit-deal.php' );
		include( 'class-autoroyal-shortcode-reservations-inventory.php' );
		include( 'class-autoroyal-shortcode-submit-reservation.php' );
		include( 'class-autoroyal-shortcode-reports.php' );
		include( 'class-autoroyal-shortcode-cars.php' );
		include( 'class-autoroyal-shortcode-cars-rent.php' );
		include( 'class-autoroyal-shortcode-featured-cars.php' );
		include( 'class-autoroyal-shortcode-recent-cars.php' );
		include( 'class-autoroyal-shortcode-popular-cars.php' );
		include( 'class-autoroyal-shortcode-search-filter.php' );
		include( 'class-autoroyal-shortcode-financial-application.php' );
		include( 'class-autoroyal-shortcode-pre-qualify-form.php' );
		include( 'class-autoroyal-shortcode-test-drive-form.php' );
		include( 'class-autoroyal-shortcode-trade-in-form.php' );
		include( 'class-autoroyal-shortcode-account-funds.php' );
		include( 'class-autoroyal-shortcode-login.php' );
		include( 'class-autoroyal-shortcode-my-favorite.php' );
		include( 'class-autoroyal-shortcode-notifications.php' );
		include( 'class-autoroyal-shortcode-upcoming-payments.php' );
		include( 'class-autoroyal-shortcode-filter-subscriptions.php' );
		include( 'class-autoroyal-shortcode-dealer.php' );
		include( 'class-autoroyal-shortcode-all-dealers.php' );
		include( 'class-autoroyal-shortcode-settings.php' );

		// Init classes
		$this->vehicle_post_types = new autoroyal_Car_Post_Type();
		$this->vehicle_post_meta = new autoroyal_Car_Post_Type_MetaBoxes();

		$this->lead_post_types = new autoroyal_Lead_Post_Type();
		$this->lead_post_meta = new autoroyal_Lead_Post_Type_MetaBoxes();

		$this->deal_post_types = new autoroyal_Deal_Post_Type();
		$this->deal_post_meta = new autoroyal_Deal_Post_Type_MetaBoxes();

		$this->deal_post_types = new autoroyal_Rent_Post_Type();
		$this->deal_post_meta = new autoroyal_Rent_Post_Type_MetaBoxes();

		$this->filter_post_types = new autoroyal_Filter_Post_Type();

		// Switch theme
		add_action( 'after_switch_theme', array( $this->vehicle_post_types, 'register_vehicle_post_type' ), 11 );
		add_action( 'after_switch_theme', array( $this->lead_post_types, 'register_lead_post_type' ), 11 );
		add_action( 'after_switch_theme', array( $this->deal_post_types, 'register_deal_post_type' ), 11 );
		add_action( 'after_switch_theme', 'flush_rewrite_rules', 15 );

		// Start Cron Jobs
		function autoroyal_add_cron_recurrence_interval( $schedules ) {
 
		    $schedules['autoroyal_every_five_minutes'] = array(
		            'interval'  => 300,
		            'display'   => __( 'Every 5 Minutes', 'autoroyal' )
		    );
		     
		    return $schedules;
		}
		add_filter( 'cron_schedules', 'autoroyal_add_cron_recurrence_interval' );

		if ( ! wp_next_scheduled( 'autoroyal_payments_cron_jobs' ) ) {
		    wp_schedule_event( time(), 'autoroyal_every_five_minutes', 'autoroyal_payments_cron_jobs' );
		}
		add_action( 'autoroyal_payments_cron_jobs', array( $this, 'autoroyal_payments_cron_jobs' ) );


		//Allow Dealers to Add Media
		add_action('admin_init', 'allow_dealer_uploads');
		 
		function allow_dealer_uploads() {
		    $dealer = get_role('dealer');
		  	// author caps
		  	$dealer->add_cap('edit_published_posts');
		  	$dealer->add_cap('delete_posts');
		  	$dealer->add_cap('edit_posts');
		  	$dealer->add_cap('upload_files');
		  	// editor caps
		  	$dealer->add_cap('edit_others_posts');
		  	$dealer->add_cap('edit_published_pages');
		}

		register_activation_hook(__FILE__, 'autoroyal_activation');
		function autoroyal_activation() {
		    deactivate_plugins('autoroyal_lite/autoroyal.php');
		}

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - autoroyal_Loader. Orchestrates the hooks of the plugin.
	 * - autoroyal_i18n. Defines internationalization functionality.
	 * - autoroyal_Admin. Defines all hooks for the admin area.
	 * - autoroyal_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.2
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-autoroyal-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-autoroyal-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/admin/class-autoroyal-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/public/class-autoroyal-public.php';

		$this->loader = new autoroyal_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the autoroyal_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.2
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new autoroyal_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.2
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new autoroyal_Admin( $this->get_autoroyal(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );


	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.2
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new autoroyal_Public( $this->get_autoroyal(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.2
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.2
	 * @return    string    The name of the plugin.
	 */
	public function get_autoroyal() {
		return $this->autoroyal;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.2
	 * @return    autoroyal_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.2
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

	/**
	 * Cron Featured & Promoted Listings
	 */
	public function autoroyal_payments_cron_jobs() {
		global $wpdb;

		/**
		 * Featured Listings
		 */
		$featured_cron_jobs = array();
		$featured_cron_jobs = get_option( 'featured_cron_jobs' );
		$featured_cron_jobs[] = date("Y-m-d H:i:s");
		update_option( 'featured_cron_jobs', $featured_cron_jobs );

		// Change status to expired
		$search_args = array(
			'post_type'           => 'vehicle',
			'posts_per_page'      => -1,
			'post_status'         => array( 'publish' ),
			'meta_query' => array(
			    array(
			        'key' => '_paid_featured',
			        'value' => 1
			    )
			)
		);

		$cars_query = new WP_Query( $search_args );
		if ( $cars_query->have_posts() ) :

			while ( $cars_query->have_posts() ) : $cars_query->the_post();

				$car_id = get_the_ID();
				$user_id = get_the_author_meta( 'ID' );
				$user_meta = get_userdata($user_id); 
				$user_roles = $user_meta->roles; 

				$now = strtotime(date("Y-m-d H:i:s"));
				$next_payment = get_post_meta( $car_id, '_featured_next_payment', true );

				$amount = get_option( 'autoroyal_featured_listing_price' );

				if( $now >= $next_payment ) {

					if( !in_array("administrator", $user_roles) ) {

						if( autoroyal_user_sufficient_funds( $user_id, $amount ) ) {

							update_post_meta( $car_id, '_promoted', 0 );
							update_post_meta( $car_id, '_featured', 1 );
							update_post_meta( $car_id, '_paid_featured', 1 );
							autoroyal_update_menu_order( $car_id, '_featured', 1 );

							$new_date_payment = strtotime("+ 1 day");
							update_post_meta( $car_id, '_featured_next_payment', $new_date_payment );

							if ( is_woocommerce_active() ) {

								autoroyal_WC_Account_Funds::remove_funds( $user_id, $amount );

							}

							autoroyal_user_save_payout( $user_id, $amount, $car_id, 'Featured' );

						} else {

							update_post_meta( $car_id, '_promoted', 0 );
							update_post_meta( $car_id, '_featured', 0 );
							update_post_meta( $car_id, '_paid_featured', 0 );
							autoroyal_update_menu_order( $car_id, '_featured', 0 );

							update_post_meta( $car_id, '_featured_next_payment', '' );

						}

					} else {

						update_post_meta( $car_id, '_promoted', 0 );
						update_post_meta( $car_id, '_featured', 1 );
						update_post_meta( $car_id, '_paid_featured', 1 );
						autoroyal_update_menu_order( $car_id, '_featured', 1 );

						$new_date_payment = strtotime("+ 1 day");
						update_post_meta( $car_id, '_featured_next_payment', $new_date_payment );

					}

				}

			endwhile;

		endif; wp_reset_postdata();

		/**
		 * Promoted Listings
		 */
		$promoted_cron_jobs = array();
		$promoted_cron_jobs = get_option( 'promoted_cron_jobs' );
		$promoted_cron_jobs[] = date("Y-m-d H:i:s");
		update_option( 'promoted_cron_jobs', $promoted_cron_jobs );

		// Change status to expired
		$search_args = array(
			'post_type'           => 'vehicle',
			'posts_per_page'      => -1,
			'post_status'         => array( 'publish' ),
			'meta_query' => array(
			    array(
			        'key' => '_promoted',
			        'value' => 1
			    )
			)
		);

		$cars_query = new WP_Query( $search_args );
		if ( $cars_query->have_posts() ) :

			while ( $cars_query->have_posts() ) : $cars_query->the_post();

				$car_id = get_the_ID();
				$user_id = get_the_author_meta( 'ID' );
				$user_meta = get_userdata($user_id); 
				$user_roles = $user_meta->roles;

				$now = strtotime(date("Y-m-d H:i:s"));
				$next_payment = get_post_meta( $car_id, '_promoted_next_payment', true );

				$amount = get_option( 'autoroyal_promoted_listing_price' );

				if( $now >= $next_payment ) {

					if( !in_array("administrator", $user_roles) ) {

						if( autoroyal_user_sufficient_funds( $user_id, $amount ) ) {

							update_post_meta( $car_id, '_promoted', 1 );

							$time = current_time('mysql');

							$my_post = array(
							  	'ID' => $car_id,
							  	'post_status' => 'publish',
							  	'post_date'     => $time,
				        		'post_date_gmt' => get_gmt_from_date( $time )
						  	);

						  	wp_update_post( $my_post );

							$new_date_payment = strtotime("+ 1 day");
							update_post_meta( $car_id, '_promoted_next_payment', $new_date_payment );

							if ( is_woocommerce_active() ) {

								autoroyal_WC_Account_Funds::remove_funds( $user_id, $amount );

							}

							autoroyal_user_save_payout( $user_id, $amount, $car_id, 'Promoted' );

						} else {

							update_post_meta( $car_id, '_promoted', 0 );
							update_post_meta( $car_id, '_promoted_next_payment', '' );

						}

					} else {

						update_post_meta( $car_id, '_promoted', 1 );

						$time = current_time('mysql');

						$my_post = array(
						  	'ID' => $car_id,
						  	'post_status' => 'publish',
						  	'post_date'     => $time,
			        		'post_date_gmt' => get_gmt_from_date( $time )
					  	);

					  	wp_update_post( $my_post );

						$new_date_payment = strtotime("+ 1 day");
						update_post_meta( $car_id, '_promoted_next_payment', $new_date_payment );
						
					}

				}

			endwhile;

		endif; wp_reset_postdata();
	}

}

// autoroyal Acount Funds Functions
/**
 * Required functions
 */
if ( ! function_exists( 'woothemes_queue_update' ) ) {
	require_once( 'accound-funds/class-autoroyal-wc-functions.php' );
}

if ( is_woocommerce_active() ) {

	/**
	 * autoRoyal_WC_Account_Funds
	 */
	class autoroyal_WC_Account_Funds {

		/**
		 * Plugin's version.
		 *
		 * @var string
		 */

		/**
		 * Constructor
		 */
		public function __construct() {

			add_action( 'plugins_loaded', array( $this, 'init_early' ) );
			add_action( 'init', array( $this, 'init' ) );
			add_action( 'plugins_loaded', array( $this, 'gateway_init' ), 0 );
			add_action( 'init', array( $this, 'admin_init' ) );
			add_filter( 'woocommerce_email_classes', array( $this, 'add_email_classes' ), 99 );
			add_filter( 'woocommerce_payment_gateways', array( $this, 'register_gateway' ) );
			add_filter( 'woocommerce_data_stores', array( $this, 'add_data_stores' ) );
			
		}

		/**
		 * Classes that need to be loaded early.
		 *
		 * @since 2.0.12
		 */
		public function init_early() {
			include_once( 'accound-funds/class-autoroyal-account-funds-my-account.php' );
		}

		/**
		 * Load classes
		 */
		public function init() {
			include_once( 'accound-funds/class-autoroyal-product-deposit.php' );
			include_once( 'accound-funds/class-autoroyal-product-topup.php' );
			include_once( 'accound-funds/class-autoroyal-account-funds-cart-manager.php' );
			include_once( 'accound-funds/class-autoroyal-account-funds-deposits-manager.php' );
			include_once( 'accound-funds/class-autoroyal-account-funds-order-manager.php' );
			//include_once( 'accound-funds/class-autoroyal-account-funds-integration.php' );
			//include_once( 'accound-funds/class-autoroyal-account-funds-shortcodes.php' );
		}

		/**
		 * Init Gateway
		 */
		public function gateway_init() {
			if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
				return;
			}
			include_once( 'accound-funds/class-autoroyal-gateway-account-funds.php' );
		}

		/**
		 * Load admin
		 */
		public function admin_init() {
			if ( is_admin() ) {
				include_once( 'accound-funds/class-autoroyal-account-funds-admin.php' );
				include_once( 'accound-funds/class-autoroyal-account-funds-admin-product.php' );
				include_once( 'accound-funds/class-autoroyal-account-funds-reports.php' );
			}
		}

		/**
		 *  Add email to the list of emails WooCommerce should load.
		 */
		public function add_email_classes( $email_classes ) {
			include_once( 'accound-funds/class-autoroyal-account-funds-email-account-funds-increase.php' );
			$email_classes['autoroyal_WC_Account_Funds_Email_Account_Funds_Increase'] = new autoroyal_WC_Account_Funds_Email_Account_Funds_Increase();
			return $email_classes;
		}

		/**
		 * Activation
		 */
		public function activate() {
			if ( ! class_exists( 'WC_Account_Funds_Installer' ) ) {
				include_once( 'accound-funds/class-autoroyal-account-funds-installer.php' );
			}
			WC_Account_Funds_Installer::install( $this->version );
			WC_Account_Funds_Installer::flush_rewrite_rules();
		}

		/**
		 * Get a users funds amount
		 * @param  int  $user_id
		 * @param  boolean $formatted
		 * @return string
		 */
		public static function get_account_funds( $user_id = null, $formatted = true, $exclude_order_id = 0 ) {
			$user_id = $user_id ? $user_id : get_current_user_id();

			if ( $user_id ) {
				$funds = max( 0, get_user_meta( $user_id, 'account_funds', true ) );

				// Account for pending orders
				$orders_with_pending_funds = get_posts( array(
					'numberposts' => -1,
					'post_type'   => 'shop_order',
					'post_status' => array_keys( wc_get_order_statuses() ),
					'fields'      => 'ids',
					'meta_query'  => array(
						array(
							'key'   => '_customer_user',
							'value' => $user_id
						),
						array(
							'key'   => '_funds_removed',
							'value' => '0',
						),
						array(
							'key'     => '_funds_used',
							'value'   => '0',
							'compare' => '>'
						)
					)
				) );

				foreach ( $orders_with_pending_funds as $order_id ) {
					if ( null !== WC()->session && ! empty( WC()->session->order_awaiting_payment ) && $order_id == WC()->session->order_awaiting_payment ) {
						continue;
					}
					if ( $exclude_order_id === $order_id ) {
						continue;
					}
					$funds = $funds - floatval( get_post_meta( $order_id, '_funds_used', true ) );
				}
			} else {
				$funds = 0;
			}

			return $formatted ? wc_price( $funds ) : $funds;
		}

		/**
		 * Add funds to user account
		 * @param int $customer_id
		 * @param float $amount
		 */
		public static function add_funds( $customer_id, $amount ) {
			$funds = get_user_meta( $customer_id, 'account_funds', true );
			$funds = $funds ? $funds : 0;
			$funds += floatval( $amount );
			update_user_meta( $customer_id, 'account_funds', $funds );
		}

		/**
		 * Remove funds from user account
		 * @param int $customer_id
		 * @param float $amount
		 */
		public static function remove_funds( $customer_id, $amount ) {
			$funds = get_user_meta( $customer_id, 'account_funds', true );
			$funds = $funds ? $funds : 0;
			$funds = $funds - floatval( $amount );
			update_user_meta( $customer_id, 'account_funds', max( 0, $funds ) );
		}

		/**
		 * Register the gateway for use
		 */
		public function register_gateway( $methods ) {
			$methods[] = 'autoroyal_WC_Gateway_Account_Funds';
			return $methods;
		}

		/**
		 * Add AF-related data stores.
		 *
		 * @param array $data_stores Data stores.
		 *
		 * @return array Data stores.
		 */
		public function add_data_stores( $data_stores ) {
			if ( ! class_exists( 'autoroyal_WC_Product_Topup_Data_Store' ) ) {
				require_once( 'accound-funds/class-autoroyal-product-topup-data-store.php' );
			}

			$data_stores['product-topup'] = 'autoroyal_WC_Product_Topup_Data_Store';

			return $data_stores;
		}

	}

	new autoroyal_WC_Account_Funds();

}
